from django.utils import timezone
from django.db import models
from employer_profile.models.profile import Profile
import uuid
from enum import Enum

class CompanyTypeEnum(Enum):
    PLUTON_ADVERTISING = 'PLUTON ADVERTISING'
    ORYXMEDIA = 'ORYXMEDIA'

    @classmethod
    def choices(cls):
        return [(tag.value, tag.name.title()) for tag in cls]

class TotalMarketing(Enum):
    NEW = 'NEW'
    EXISTING = 'EXISTING'

    @classmethod
    def choices(cls):
        return [(tag.value, tag.name.title()) for tag in cls]

class TYPEOFWORK(Enum):
    DIGITAL_PRINTING = 'DIGITAL PRINTING'
    OFFSET_PRINTING = 'OFFSET PRINTING'
    GIFTS = 'GIFTS'
    SIGNAGES = 'SIGNAGES'
    BRANDING = 'BRANDING'
    REPROGRAPHICS = 'REPROGRAPHICS'
    OTHERS = 'OTHERS'

    @classmethod
    def choices(cls):
        return [(tag.value, tag.value) for tag in cls]


class MarketVisit(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    profiles = models.ForeignKey(Profile, on_delete=models.CASCADE, related_name='reports')
    date = models.DateField(auto_now_add=True)
    created_at = models.DateTimeField(default=timezone.now, null=True, blank=True)
    company_title = models.CharField(
        max_length=40,
        choices=CompanyTypeEnum.choices()
    )

    def __str__(self):
        return f"Market Visit on {self.date} by {self.profiles}"

    class Meta:
        ordering = ['-date']


class Enquiries(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    market = models.ForeignKey(MarketVisit, on_delete=models.CASCADE, related_name='enquiries')
    customer_name = models.CharField(max_length=200, null=True, blank=True)
    type_of_work = models.CharField(
        max_length=32,
        null=True,
        blank=True,
        choices=TYPEOFWORK.choices()
    )
    descriptions = models.TextField()
    created_at = models.DateTimeField(default=timezone.now, null=True, blank=True)

    def __str__(self):
        return str(self.customer_name)

    class Meta:
        ordering = ['-created_at']


class QuotesSubmitted(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    market = models.ForeignKey(MarketVisit, on_delete=models.CASCADE, related_name='quotes_submitted')
    customer_name = models.CharField(max_length=200, null=True, blank=True)
    type_of_work = models.CharField(
        max_length=32,
        null=True,
        blank=True,
        choices=TYPEOFWORK.choices()
    )
    Quote_no = models.CharField(max_length=30, null=True, blank=True)
    quote_amount = models.CharField(max_length=50, null=True, blank=True)
    remarks = models.TextField()
    created_at = models.DateTimeField(default=timezone.now, null=True, blank=True)

    def __str__(self):
        return str(self.customer_name)

    class Meta:
        ordering = ['-created_at']


class ConfirmedWork(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    market = models.ForeignKey(MarketVisit, on_delete=models.CASCADE, related_name='confirmed_work')
    customer_name = models.CharField(max_length=200, null=True, blank=True)
    type_of_work = models.CharField(
        max_length=32,
        null=True,
        blank=True,
        choices=TYPEOFWORK.choices()
    )
    so_no = models.CharField(max_length=30, null=True, blank=True)
    so_amount = models.CharField(max_length=50, null=True, blank=True)
    remarks = models.TextField()
    created_at = models.DateTimeField(default=timezone.now, null=True, blank=True)

    def __str__(self):
        return str(self.customer_name)

    class Meta:
        ordering = ['-created_at']


class InvoiceWork(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    market = models.ForeignKey(MarketVisit, on_delete=models.CASCADE, related_name='invoiced_work')
    customer_name = models.CharField(max_length=200, null=True, blank=True)
    type_of_work = models.CharField(
        max_length=32,
        null=True,
        blank=True,
        choices=TYPEOFWORK.choices()
    )
    invoice_no = models.CharField(max_length=30, null=True, blank=True)
    invoice_amount = models.CharField(max_length=50, null=True, blank=True)
    remarks = models.TextField()
    created_at = models.DateTimeField(default=timezone.now, null=True, blank=True)

    def __str__(self):
        return f"{self.customer_name} - ₹{self.invoice_amount}"

    class Meta:
        ordering = ['-created_at']


class PaymentCollected(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    market = models.ForeignKey(MarketVisit, on_delete=models.CASCADE, related_name='payments_collected')
    customer_name = models.CharField(max_length=200, null=True, blank=True)
    invoice_no = models.CharField(max_length=30, null=True, blank=True)
    rcvd_amount = models.CharField(max_length=20, null=True, blank=True)
    remarks = models.TextField()
    created_at = models.DateTimeField(default=timezone.now, null=True, blank=True)

    def __str__(self):
        return f"{self.customer_name} - ₹{self.rcvd_amount}"

    class Meta:
        ordering = ['-created_at']


class Marketing(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    market = models.ForeignKey(MarketVisit, on_delete=models.CASCADE, related_name='total_marketing')
    client_name = models.CharField(max_length=100)
    status = models.CharField(max_length=50, choices=TotalMarketing.choices(), default=TotalMarketing.NEW.value)
    email = models.EmailField(max_length=100, null=True, blank=True)
    contact_no = models.CharField(max_length=20, null=True, blank=True)
    created_at = models.DateTimeField(default=timezone.now, null=True, blank=True)

    def __str__(self):
        return f"{self.client_name} - {self.status}"

    class Meta:
        ordering = ['-created_at']
